<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SaleVoucher;
use App\Models\SaleVoucheritem;
use App\Models\ProductBuyingLog;
use App\Models\Productvariant;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Str;



class SaleVoucherController extends Controller
{
    
    public function new_sale_voucher(Request $request)
    {
        $validated = $request->validate([
           
            'business_id' => 'required|integer',
            'sale_item' => 'required|array',
            
        ]);

        $voucherCode = $this->generateUniqueVoucherCode();

        $businessId = $request->business_id;
        $products = $request->sale_item;
        $results = [];
    
        DB::beginTransaction();
    
        try {
            // ✅ Check stock before selling
            $notEnoughStock = [];
            if (!$this->hasEnoughStockForAll($products, $businessId, $notEnoughStock)) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient stock for one or more product variants.',
                    'not_enough_stock' => $notEnoughStock
                ], 400);
            }

        $new_sale = new SaleVoucher();
        $new_sale->voucher_code = $voucherCode;
        $new_sale->customer_id = $request->customer_id ?? null;
        $new_sale->sale_date = Carbon::now()->toDateTimeString();
        $new_sale->business_id = $request->business_id;
        $new_sale->payment_status = $request->payment_status;
        $new_sale->discount_amount = $request->discount_amount;
        $new_sale->tax_amount = $request->tax_amount;
        $new_sale->remark = $request->remark;
        $new_sale->save();

        $lastsale = SaleVoucher::latest()->first();


        foreach ($request->sale_item as $item) {

            $saleitem = new SaleVoucheritem();
            $saleitem->sale_voucher_id = $lastsale->id;
            $saleitem->product_variant_id = $item['product_variant_id'];
            $saleitem->quantity = $item['quantity'];
            $saleitem->sold_price = $item['sold_price'];
            $saleitem->business_id = $request->business_id;
            $saleitem->save();

        }

    
            // ✅ Proceed to sell each product
            foreach ($products as $product) {
                $variantId = $product['product_variant_id'];
                $sellQty = $product['quantity'];
                $sellPrice = $product['sold_price'];
               
    
                $this->processVariantSale($variantId, $sellQty, $sellPrice, $businessId);
    
                $remainingQty = ProductBuyingLog::where('product_variant_id', $variantId)
                    ->where('business_id', $businessId)
                    ->sum(DB::raw('purchase_quantity - sold_quantity'));

                    $variant = Productvariant::find($variantId);
                    $variantName = $variant ? $variant->name : 'Unknown';
            
    
                $results[] = [
                    'product_variant_id' => $variantId,
                    'product_variant_name' => $variantName,
                    'sold_quantity' => $sellQty,
                    'remaining_quantity' => $remainingQty
                ];
            }
    
            DB::commit();
    
            return response()->json([
                'success' => true,
                'message' => 'All products sold successfully.',
                'results' => $results
            ]);
    
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Sale failed.',
                'error' => $e->getMessage()
            ], 500);
        }

    }

    private function hasEnoughStockForAll($products, $businessId, &$notEnoughStock)
{
    $notEnoughStock = [];

    foreach ($products as $product) {
        $variantId = $product['product_variant_id'];
        $sellQty = $product['quantity'];

        $availableQty = ProductBuyingLog::where('product_variant_id', $variantId)
            ->where('business_id', $businessId)
            ->sum(DB::raw('purchase_quantity - sold_quantity'));

        if ($availableQty < $sellQty) {

            $variant = Productvariant::find($variantId);
            $variantName = $variant ? $variant->name : 'Unknown';

            $notEnoughStock[] = [
                'product_variant_id' => $variantId,
                'product_variant_name' => $variantName,
                'available_quantity' => $availableQty,
                'requested_quantity' => $sellQty
            ];
        }
    }

    return empty($notEnoughStock); // ✅ true = all okay, ❌ false = not enough stock
}

    private function processVariantSale($variantId, $sellQty, $sellPrice, $businessId)
{
    $remainingToSell = $sellQty;

    $stocks = ProductBuyingLog::where('product_variant_id', $variantId)
        ->where('business_id', $businessId)
        ->whereColumn('purchase_quantity', '>', 'sold_quantity')
        ->orderBy('buy_date')
        ->orderByRaw('CASE WHEN expired_date IS NULL THEN 1 ELSE 0 END, expired_date ASC')
        ->get();

    foreach ($stocks as $stock) {
        if ($remainingToSell <= 0) break;

        $availableQty = $stock->purchase_quantity - $stock->sold_quantity;
        $deductQty = min($availableQty, $remainingToSell);

        $stock->sold_quantity += $deductQty;
        $stock->status = $stock->sold_quantity >= $stock->purchase_quantity ? 1 : 0;
        $stock->updated_at = Carbon::now();
        $stock->save();

        $remainingToSell -= $deductQty;
    }

    return $remainingToSell <= 0;
}


    function generateUniqueVoucherCode($length = 8){
    do {
        $code = strtoupper(Str::random($length)); // e.g., "8GH7K2QM"
    } while (SaleVoucher::where('voucher_code', $code)->exists());

    return $code;
      }

      

public function getSales_vouchers(Request $request)
{

    $validated = $request->validate([
           
        'business_id' => 'required|integer',
        'page' => 'required|integer'
        
    ]);

        $page = $request->input('page', 1); // default page 1
        $limit = $request->input('limit', 10); // default 10 items per page
        $offset = ($page - 1) * $limit; // calculate offset from page

     // Step 1: Get paginated sale vouchers
     $saleVouchers = DB::table('sale_vouchers')
     ->where('business_id', $request->business_id)
     ->orderBy('created_at', 'desc')
     ->offset($offset)
     ->limit($limit)
     ->get();

 // Extract voucher IDs
 $voucherIds = $saleVouchers->pluck('id');

 // Step 2: Get sale items joined with product variants
 $items = DB::table('sale_voucheritems')
     ->join('productvariants', 'sale_voucheritems.product_variant_id', '=', 'productvariants.id')
     ->whereIn('sale_voucheritems.sale_voucher_id', $voucherIds)
     ->select(
         'sale_voucheritems.id as item_id',
         'sale_voucheritems.sale_voucher_id',
         'sale_voucheritems.product_variant_id',
         'sale_voucheritems.quantity',
         'sale_voucheritems.sold_price',
          DB::raw('sale_voucheritems.quantity * sale_voucheritems.sold_price as total_item_amount'),
         'productvariants.name as variant_name'
     )
     ->get()
     ->groupBy('sale_voucher_id'); // Group items by voucher

 // Step 3: Attach items to each voucher
 $sales = $saleVouchers->map(function ($voucher) use ($items) {
     $voucherItems = $items[$voucher->id] ?? [];
     $voucherTotalAmount = collect($voucherItems)->sum('total_item_amount');
     $voucher->voucher_total_amount = $voucherTotalAmount;
     $balance = $voucherTotalAmount - ($voucher->discount_amount ?? 0) + ($voucher->tax_amount ?? 0);
     $voucher->balance = $balance;

     $voucher->items = $items[$voucher->id] ?? [];
   
     return $voucher;
 });


    return response()->json([
        'message' => 'Sale  query successfully', 
        'status' => 200,
        'data' => $sales,
       
    ]);

}



public function getTodaySales_vouchers(Request $request)
{

    $validated = $request->validate([
           
        'business_id' => 'required|integer',
        'page' => 'required|integer'
        
    ]);

        $page = $request->input('page', 1); // default page 1
        $limit = $request->input('limit', 10); // default 10 items per page
        $offset = ($page - 1) * $limit; // calculate offset from page

        $today = Carbon::today()->toDateString();


     // Step 1: Get paginated sale vouchers
     $saleVouchers = DB::table('sale_vouchers')
     ->where('business_id', $request->business_id)
     ->whereDate('sale_date', $today)
     ->orderBy('created_at', 'desc')
     ->offset($offset)
     ->limit($limit)
     ->get();

 // Extract voucher IDs
 $voucherIds = $saleVouchers->pluck('id');

 // Step 2: Get sale items joined with product variants
 $items = DB::table('sale_voucheritems')
     ->join('productvariants', 'sale_voucheritems.product_variant_id', '=', 'productvariants.id')
     ->whereIn('sale_voucheritems.sale_voucher_id', $voucherIds)
     ->select(
         'sale_voucheritems.id as item_id',
         'sale_voucheritems.sale_voucher_id',
         'sale_voucheritems.product_variant_id',
         'sale_voucheritems.quantity',
         'sale_voucheritems.sold_price',
          DB::raw('sale_voucheritems.quantity * sale_voucheritems.sold_price as total_item_amount'),
         'productvariants.name as variant_name'
     )
     ->get()
     ->groupBy('sale_voucher_id'); // Group items by voucher

 // Step 3: Attach items to each voucher
 $sales = $saleVouchers->map(function ($voucher) use ($items) {
     $voucherItems = $items[$voucher->id] ?? [];
     $voucherTotalAmount = collect($voucherItems)->sum('total_item_amount');
     $voucher->voucher_total_amount = $voucherTotalAmount;
     $balance = $voucherTotalAmount - ($voucher->discount_amount ?? 0) + ($voucher->tax_amount ?? 0);
     $voucher->balance = $balance;

     $voucher->items = $items[$voucher->id] ?? [];
   
     return $voucher;
 });

 $todayTotalSaleAmount = $sales->sum('voucher_total_amount');


    return response()->json([
        'message' => 'Sale  query successfully', 
        'status' => 200,
        'totalsale_amount' => $todayTotalSaleAmount,
        'data' => $sales,
       
    ]);

}






}
